﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.MappingService.Interface;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public abstract class MapperRelatedBase : MapperBase, IEntityChild
    {
        private List<Entity> _modifyEntityList;

        public Entity ParentalReference { get; set; }
        
        protected List<Entity> ModifyEntityList => _modifyEntityList ?? (_modifyEntityList = new List<Entity>());

        protected MapperRelatedBase(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public abstract Task<Entity> MapInsert<T>(T entity, Entity parent);

        public abstract Task<Entity> MapUpdate<T>(T entity, Entity parent);

        public abstract void AddChildrenToProvider(IList<Entity> entities, Entity parent);

        public abstract IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent);

        public async Task<IEnumerable<Entity>> InsertMapperEntity<T>(
            IEnumerable<T> schemaList, 
            Entity parent, 
            bool skipProviderRelationship = false,
            bool forVaNetwork = false)
        {
            ForVaNetwork = forVaNetwork;
            ResetContext();

            var list = new List<Entity>();
            if (schemaList != null)
            {
                foreach (var item in schemaList)
                {
                    var child = await MapInsert(item, parent);
                    if (child != null)
                    {
                        list.Add(child);
                        ModifyEntityList.Add(child);
                    }
                }

                if (!skipProviderRelationship && ModifyEntityList.Count > 0 && parent.GetType() == typeof(Account)) AddChildrenToProvider(list, parent);
            }
            return list;
        }

        public async Task<IEnumerable<Entity>> UpdateMapperEntity<T>(
            IEnumerable<T> schemaList, 
            Entity parent, 
            bool skipProviderRelationship = false,
            bool forVaNetwork = false)
        {
            ForVaNetwork = forVaNetwork;

            var list = new List<Entity>();
            if (schemaList != null)
            {
                foreach (var item in schemaList)
                {
                    var child = await MapUpdate(item, parent);
                    if (child != null)
                    {
                        list.Add(child);
                        ModifyEntityList.Add(child);
                    }
                }

                if (!skipProviderRelationship && ModifyEntityList.Count > 0 && parent.GetType() == typeof(Account)) AddChildrenToProvider(list, parent);
            }

            return list;
        }

        public IEnumerable<SetStateRequest> DeleteMapperEntity<T>(IList<T> schemaList, Entity entity)
        {
            return MapDelete(schemaList, entity);
        }

        public void AddModifiedEntity(Entity entity)
        {
            ModifyEntityList.Add(entity);
        }

        public void SetContext(PpmsContext context)
        {
            Context = context;
        }

        protected void AssociateRelatedEntities(Entity parent, Entity entity, string relationship)
        {
            AssociateRelatedEntities(parent, new List<Entity>() { entity }, relationship);
        }

        protected void AssociateRelatedEntities(Entity parent, IEnumerable<Entity> entities, string relationship)
        {
            if (entities != null)
            {
                bool isAddingRelationship;
                // Make sure parent entity is being tracked
                if (!Context.IsAttached(parent)) Context.AddObject(parent);

                EntityCollection children = null;
                var account = (Account)ParentalReference;
                if (account != null)
                {
                    var rel = new Relationship(relationship);
                    if (account.RelatedEntities.Contains(rel))
                        children = account.RelatedEntities[rel];
                }

                foreach (var item in entities)
                {
                    isAddingRelationship = true;

                    // If parental reference exists, make sure the child is not associated parent
                    if (children != null && children.Entities != null 
                        && children.Entities.Any(a => a.Id == item.Id))
                    {
                        isAddingRelationship = false;
                        if (!Context.IsAttached(item))
                        {
                            Context.Attach(item);
                            Context.UpdateObject(item);
                        }
                    }

                    if (isAddingRelationship) Context.AddRelatedObject(parent, new Relationship(relationship), item);
                }
            }
        }

        protected void AddRelatedEntities(Entity item, string parentRelationship, string childRelationship = null)
        {
            var relationship = new Relationship(parentRelationship);

            if (item.RelatedEntities != null && item.RelatedEntities.ContainsKey(relationship))
            {
                var children = item.RelatedEntities[relationship];
                if (children != null && children.Entities != null && children.Entities.Count > 0)
                {
                    foreach (var entity in children.Entities)
                    {
                        if (!Context.IsAttached(entity))
                        {
                            Context.AddRelatedObject(item, relationship, entity);
                            if (!string.IsNullOrEmpty(childRelationship))
                            {
                                AddRelatedEntities(entity, childRelationship);
                            }
                        }
                    }
                }
            }
        }

    }
}